--[[
   Copyright (c) The OpenRA Developers and Contributors
   This file is part of OpenRA, which is free software. It is made
   available to you under the terms of the GNU General Public License
   as published by the Free Software Foundation, either version 3 of
   the License, or (at your option) any later version. For more
   information, see COPYING.
]]
InsertionHelicopterType = "tran.insertion"
InsertionPath = { InsertionEntry.Location, InsertionLZ.Location }
ExtractionHelicopterType = "tran.extraction"
ExtractionPath = { SouthReinforcementsPoint.Location, ExtractionLZ.Location }
JeepReinforcements = { "jeep", "jeep" }
TanyaReinforcements = { "e7.noautotarget" }
EinsteinType = "einstein"
FlareType = "flare"
CruisersReinforcements = { "ca", "ca", "ca", "ca" }
OpeningAttack = { Patrol1, Patrol2, Patrol3, Patrol4 }
Responders = { Response1, Response2, Response3, Response4, Response5 }
LabGuardsTeam = { LabGuard1, LabGuard2, LabGuard3 }

SendInsertionHelicopter = function()
	local passengers = Reinforcements.ReinforceWithTransport(GoodGuy, InsertionHelicopterType,
		TanyaReinforcements, InsertionPath, { InsertionEntry.Location })[2]
	local tanya = passengers
	Trigger.OnAllKilled(tanya, TanyaKilledInAction)
	Trigger.AfterDelay(DateTime.Seconds(4), function()
		Media.DisplayMessage(UserInterface.GetFluentMessage("tanya-rules-of-engagement"), UserInterface.GetFluentMessage("tanya"))
	end)
end

SendJeeps = function()
	Reinforcements.Reinforce(GoodGuy, JeepReinforcements, InsertionPath, DateTime.Seconds(2))
	Media.PlaySpeechNotification(All, "ReinforcementsArrived")
end

RunInitialActivities = function()
	SendInsertionHelicopter()

	Utils.Do(OpeningAttack, function(a)
		IdleHunt(a)
	end)

	Trigger.OnKilled(Patrol3, function()
		if not Civilian1.IsDead then
			Civilian1.Move(CivMove.Location)
		end
	end)

	Trigger.OnKilled(BarrelPower, function()
		if not Civilian2.IsDead then
			Civilian2.Move(CivMove.Location)
		end
		Utils.Do(Responders, function(r)
			if not r.IsDead then
				IdleHunt(r)
			end
		end)
	end)
end

LabGuardsKilled = function()
	CreateEinstein()

	Trigger.AfterDelay(DateTime.Seconds(2), function()
		Actor.Create(FlareType, true, { Owner = England, Location = ExtractionFlarePoint.Location })
		Media.PlaySpeechNotification(All, "SignalFlareNorth")
		SendExtractionHelicopter()
	end)

	Trigger.AfterDelay(DateTime.Seconds(10), function()
		Media.PlaySpeechNotification(All, "AlliedReinforcementsArrived")
		Actor.Create("camera", true, { Owner = Greece, Location = CruiserCameraPoint.Location })
		SendCruisers()
	end)

	Trigger.AfterDelay(DateTime.Seconds(12), function()
		for i = 0, 2 do
			Trigger.AfterDelay(DateTime.Seconds(i), function()
				Media.PlaySoundNotification(All, "AlertBuzzer")
			end)
		end
		Utils.Do(SovietArmy, function(a)
			if not a.IsDead and a.HasProperty("Hunt") then
				Trigger.OnIdle(a, a.Hunt)
			end
		end)
	end)
end

SendExtractionHelicopter = function()
	Heli = Reinforcements.ReinforceWithTransport(GoodGuy, ExtractionHelicopterType, nil, ExtractionPath)[1]
	if not Einstein.IsDead then
		Trigger.OnRemovedFromWorld(Einstein, EvacuateHelicopter)
	end
	Trigger.OnKilled(Heli, RescueFailed)
	Trigger.OnRemovedFromWorld(Heli, HelicopterGone)
end

EvacuateHelicopter = function()
	if Heli.HasPassengers then
		Heli.Move(ExtractionExitPoint.Location)
		Heli.Destroy()
	end
end

SendCruisers = function()
	local i = 1
	Utils.Do(CruisersReinforcements, function(cruiser)
		local ca = Actor.Create(cruiser, true, { Owner = England, Location = SouthReinforcementsPoint.Location + CVec.New(2 * i, 0) })
		ca.Move(Map.NamedActor("CruiserPoint" .. i).Location)
		i = i + 1
	end)
end

LabDestroyed = function()
	if not Einstein then
		RescueFailed()
	end
end

RescueFailed = function()
	Media.PlaySpeechNotification(All, "ObjectiveNotMet")
	for _, PID in ipairs(CoopPlayers) do
	PID.MarkFailedObjective(EinsteinSurviveObjective)
	end
end

TanyaKilledInAction = function()
	Media.PlaySpeechNotification(All, "ObjectiveNotMet")
	for _, PID in ipairs(CoopPlayers) do
	PID.MarkFailedObjective(TanyaSurviveObjective)
	end
end

OilPumpDestroyed = function()
	Trigger.AfterDelay(DateTime.Seconds(5), SendJeeps)
end

CiviliansKilled = function()
	for _, PID in ipairs(CoopPlayers) do
	PID.MarkFailedObjective(CivilProtectionObjective)
	end
	Media.PlaySpeechNotification(All, "ObjectiveNotMet")
	CollateralDamage = true
end

LostMate = function()
	if not Civilian2.IsDead then
		Civilian2.Panic()
	end
end

CreateEinstein = function()
	for _, PID in ipairs(CoopPlayers) do
	PID.MarkCompletedObjective(FindEinsteinObjective)
	end
	Media.PlaySpeechNotification(All, "ObjectiveMet")
	Einstein = Actor.Create(EinsteinType, true, { Location = EinsteinSpawnPoint.Location, Owner = Greece })
	Einstein.Scatter()
	Trigger.OnKilled(Einstein, RescueFailed)
	ExtractObjective = AddPrimaryObjective(Greece, "extract-einstein-helicopter")
	Trigger.AfterDelay(DateTime.Seconds(1), function() Media.PlaySpeechNotification(All, "TargetFreed") end)
end

HelicopterGone = function()
	if not Heli.IsDead then
		Media.PlaySpeechNotification(All, "TargetRescued")
		Trigger.AfterDelay(DateTime.Seconds(1), function()
			for _, PID in ipairs(CoopPlayers) do
			PID.MarkCompletedObjective(ExtractObjective)
			PID.MarkCompletedObjective(EinsteinSurviveObjective)
			end
			if not Greece.IsObjectiveFailed(TanyaSurviveObjective) then
				for _, PID in ipairs(CoopPlayers) do
				PID.MarkCompletedObjective(TanyaSurviveObjective)
				end
			end
			if not CollateralDamage then
				for _, PID in ipairs(CoopPlayers) do
				PID.MarkCompletedObjective(CivilProtectionObjective)
				end
			end
		end)
	end
end

SetUnitStances = function()
	Utils.Do(Map.NamedActors, function(a)
		if a.Owner == GoodGuy then
			a.Stance = "Defend"
		end
	end)
end

Tick = function()
	USSR.Resources = USSR.Resources - (0.01 * USSR.ResourceCapacity / 25)
end

WorldLoaded = function()
	Greece = Player.GetPlayer("Greece")
	England = Player.GetPlayer("England")
	USSR = Player.GetPlayer("USSR")
	
	--Mission specific Information for the Coop Base Script
	FMVExtra = "prolog.vqa"
	FMVBriefing = "ally1.vqa"
	FMVOpening = "landing.vqa"
	Enemyplayer = USSR
	--
	
	CoopInit()
	
	for _, PID in ipairs(CoopPlayers) do
		if PID ~= Greece then
			table.insert(TanyaReinforcements, "e7.noautotarget")
		end
	end

	InitObjectives(Greece)

	FindEinsteinObjective = AddPrimaryObjective(Greece, "find-einstein")
	TanyaSurviveObjective = AddPrimaryObjective(Greece, "tanya-survive")
	EinsteinSurviveObjective = AddPrimaryObjective(Greece, "einstein-survive")
	CivilProtectionObjective = AddSecondaryObjective(Greece, "protect-civilians")

	RunInitialActivities()

	Trigger.OnKilled(Lab, LabDestroyed)
	Trigger.OnKilled(OilPump, OilPumpDestroyed)

	SovietArmy = USSR.GetGroundAttackers()

	Trigger.OnAllKilled(LabGuardsTeam, LabGuardsKilled)

	CollateralDamage = false
	local civilianTeam = { Civilian1, Civilian2 }
	Trigger.OnAnyKilled(civilianTeam, CiviliansKilled)
	Trigger.OnKilled(Civilian1, LostMate)

	SetUnitStances()

	Trigger.AfterDelay(DateTime.Seconds(5), function() Actor.Create("camera", true, { Owner = Greece, Location = BaseCameraPoint.Location }) end)

	Camera.Position = InsertionLZ.CenterPosition
end
