function centraliseCamera()
	local centerPosition = WPos.New(1024 * mapWidth/2, 1024 * mapHeight/2, 0)
	Camera.Position = centerPosition
end

function getWinLossConditionID()
	local missionDescription
	if (endless) then
		missionDescription = "Do not allow " .. playerLives .. " enemies to the exit!"
	else
		missionDescription = "Do not allow " .. playerLives .. " enemies to the exit and survive " .. wavesToSurvive .. " waves!"
	end
	return humanPlayer.AddPrimaryObjective(missionDescription)
end

function inTable(value, t)
	return Utils.Any(t, function(element)
			if (element == value) then
				return true
			end
			return false
		end)
end

function filterTable(t, filter)
	local filteredTable = {}
	for i, v in ipairs(t) do
		if inTable(v.Type, filter) then
			filteredTable[#filteredTable + 1] = v
		end
	end
	if (#filteredTable == 0) then
		error("filterTable Error: Filtered table empty")
	end
	return filteredTable
end

function getSpawns()
	local spawns = filterTable(Map.NamedActors, {"wave_spawn"})
	if (#spawns == 0) then
		error("getSpawns Error: No spawns on map")
	end
	return spawns
end

function getObjectives()
	local objectives = filterTable(Map.NamedActors, {"wave_objective"})
	if (#objectives == 0) then
		error("getObjectives Error: No objectives on map")
	end
	return objectives
end

function displayGameStateText()
	local gameState
	if (endless == nil) then
		error("displayGameStateText Error: Boolean endless is nil")
	end
	if (endless) then
		gameState = "WAVE: " .. waveNumber .. "\n\n"
	else
		if (wavesToSurvive == nil) then
			error("displayGameStateText Error: Variable wavesToSurvive is nil")
		elseif (wavesToSurvive < 1) then
			error("displayGameStateText Error: Variable wavesToSurvive is out of bounds")
		end
		gameState = "WAVE: " .. waveNumber .. "/" .. wavesToSurvive .. "\n\n"
	end
	gameState = gameState .. "LIVES: ["
	for i = 1, initialPlayerLives do
		if (playerLives >= i) then
			gameState = gameState .. "|"
		else
			gameState = gameState .. " "
		end
	end
	gameState = gameState .. "]"
	UserInterface.SetMissionText(gameState)
end

function disguiseActorsAs(actors, appearance)
	for i, v in ipairs(actors) do
		v.DisguiseAsType(appearance, humanPlayer)
	end
	return actors
end

function initialiseSpawns()
	enemySpawns = makeOwnerOfActors(enemyPlayer, enemySpawns)
	enemySpawns = disguiseActorsAs(enemySpawns, spawnAppearance)
end

function addObjectiveTrigger(objective)
	Trigger.OnPassengerEntered(objective, function(objective, actor)
			objective.UnloadPassenger()
			actor.Kill()
			playerLives = playerLives - 1
			if (playerLives == 0) then
				endGame(false)
			end
			displayGameStateText()
		end)
end

function addObjectiveTriggers()
	for i, v in ipairs(enemyObjectives) do
		addObjectiveTrigger(v)
	end
end

function makeOwnerOfActors(player, actors)
	for i, v in ipairs(actors) do
		v.Owner = player
	end
	return actors
end

function initialiseObjectives()
	enemyObjectives = makeOwnerOfActors(enemyPlayer, enemyObjectives)
	enemyObjectives = disguiseActorsAs(enemyObjectives, objectiveAppearance)
	addObjectiveTriggers()
end

function getPathTiles()
	local tiles = filterTable(Map.NamedActors, {"path_tile"})
	if (#tiles == 0) then
		error("getPathTiles Error: No tiles on map")
	end
	return tiles
end

function getPathBorderTiles()
	local tiles = filterTable(Map.NamedActors, {"path_border_tile"})
	if (#tiles == 0) then
		error("getPathBorderTiles Error: No tile borders on map")
	end
	return tiles
end

function actualiseActors(actors)
	for i, v in ipairs(actors) do
		v.IsInWorld = true
	end
end

function deactualiseActors(actors)
	for i, v in ipairs(actors) do
		v.IsInWorld = false
	end
end

function actualisePath()
	actualiseActors(pathTiles)
end

function actualisePathBorders()
	actualiseActors(pathBorderTiles)
end

function deactualisePath()
	deactualiseActors(pathTiles)
end

function deactualisePathBorders()
	deactualiseActors(pathBorderTiles)
end

function attachReadyTrigger(actor)
	Trigger.OnProduction(actor, function(producer, produced)
			if (produced.Type == "ready") then
				produced.Destroy()
				exitBuildMode()
				playing = true
				enterPlayMode()
			end
		end)
end

function createBuildingActor()
	local actorPosition = enemyObjectives[1].Location
	local actor = Actor.Create("build_control", true, {Location = actorPosition, Owner = humanPlayer})
	attachReadyTrigger(actor)
	return actor
end

function enterBuildMode()
	if (playing) then
		error("enterBuildMode Error: Cannot enter build mode whilst playing")
	end
	deactualisePathBorders()
	buildingActor = createBuildingActor()
	building = true
	displayGameStateText()
end

function exitBuildMode()
	building = false
	buildingActor.Destroy()
	actualisePathBorders()
end

function getWaveAllowance()
	waveAllowance = waveNumber * 2		--OPTION: Change this to a custom expression
	return waveAllowance
end

function getPossibleEnemies(allowance)
	local possibleEnemies = {}
	for i, v in ipairs(enemies) do
		if (v.Cost <= allowance) then
			table.insert(possibleEnemies, 1, v)
		end
	end
	return possibleEnemies
end

function getMaxCostUnit(units)
	local maxCost = -1
	for i, v in ipairs(units) do
		if (v.Cost > maxCost) then
			maxCost = v.Cost
		end
	end
	return maxCost
end

function populateWave()
	waveNumber = waveNumber + 1
	local allowance = getWaveAllowance()
	local possibleEnemies = getPossibleEnemies(allowance)
	while (allowance ~= 0) do
		if (getMaxCostUnit(possibleEnemies) > allowance) then
			possibleEnemies = getPossibleEnemies(allowance)
		end
		local enemy = Utils.Random(possibleEnemies)
		table.insert(wave, 1, enemy)
		allowance = allowance - enemy.Cost
	end
end

function enterPlayMode()
	if (building) then
		error("enterPlayMode Error: Cannot enter play mode whilst building")
	end
	deactualisePath()
	populateWave()
	tickCounter = 0
	playing = true
	displayGameStateText()
end

function exitPlayMode()
	playing = false
	actualisePath()
end

function waveEmpty()
	if (#wave == 0) then
		return true
	end
	return false
end

function createUnit(unitType, unitLocation)
	local unit = Actor.Create(unitType, true, {Owner = enemyPlayer, Location = unitLocation})
	return unit
end

function getRandomObjective()
	local objective = Utils.Random(enemyObjectives)
	return objective
end

function addIdleEnterTrigger(unit, objective)
	Trigger.OnIdle(unit, function(unit)
			unit.EnterTransport(objective)
		end)
end

function findInTable(value, t)
	for i, v in ipairs(t) do
		if (v == value) then
			return i
		end
	end
	error("findInTable Error: Value not in table")
end

function addOnKilledTrigger(unit)
	Trigger.OnKilled(unit, function(unit)
			local key = findInTable(unit, aliveEnemies)
			table.remove(aliveEnemies, key)
			unit.Destroy()
		end)
end

function addEnemyTriggers(unit, objective)
	addIdleEnterTrigger(unit, objective)
	addOnKilledTrigger(unit)
end

function sendUnit(spawn)
	local unitType = table.remove(wave, 1).Type
	local unit = createUnit(unitType, spawn.Location)
	table.insert(aliveEnemies, 1, unit)
	local objective = getRandomObjective()
	addEnemyTriggers(unit, objective)
	unit.EnterTransport(objective)
end

function getRandomSpawn()
	local spawn = Utils.Random(enemySpawns)
	return spawn
end

function sendUnits()
	if (parallelUnitSending) then
		for i = 1, #enemySpawns do
			if (not waveEmpty()) then
				local spawn = enemySpawns[i]
				sendUnit(spawn)
			end
		end
	else
		local spawn = getRandomSpawn()
		sendUnit(spawn)
	end
end

function enemiesAlive()
	if (#aliveEnemies == 0) then
		return false
	end
	return true
end

function finalWave()
	if (wavesToSurvive <= waveNumber) then
		return true
	end
end

function endGame(won)
	if (won) then
		humanPlayer.MarkCompletedObjective(missionID)
	else
		humanPlayer.MarkFailedObjective(missionID)
	end
end

function Tick()
	tickCounter = tickCounter + 1
	if (playing == true) then
		if ((not waveEmpty()) and (tickCounter % sendDelay == 0) and (tickCounter >= initialSendDelay)) then
			sendUnits()
		elseif ((not enemiesAlive()) and (waveEmpty())) then
			if (not endless) then
				if (finalWave()) then
					endGame(true)
					return
				end
			end
			exitPlayMode()
			enterBuildMode()
		end
	end
end

function WorldLoaded()
	centraliseCamera()
	missionID = getWinLossConditionID()
	enemySpawns 	= getSpawns()
	initialiseSpawns()
	enemyObjectives = getObjectives()
	initialiseObjectives()
	pathTiles 				= getPathTiles()
	pathBorderTiles 		= getPathBorderTiles()
	enterBuildMode()
end

--PLAYER VARIABLES--
humanPlayer = Player.GetPlayer("Multi0")
enemyPlayer = Player.GetPlayer("Creeps")
playerLives = 10
initialPlayerLives = playerLives

--MAP VARIABLES--
mapWidth 			= 32		--OPTION: Change this to your map's width
mapHeight 			= 32		--OPTION: Change this to your map's height
enemySpawns 		= {}
spawnAppearance		= "kenn"	--OPTION: Change this to the actor type that you want the spawn to appear as (nil = no disguise)
enemyObjectives		= {}
objectiveAppearance	= "silo"	--OPTION: Change this to the actor type that you want the objective to appear as (nil = no disguise)
tickCounter = 0

--WIN LOSS CONDITION VARIABLES--
missionID = nil

--PLAY MODE VARIABLES--
playing = false
aliveEnemies = {}

--WAVE VARIABLES--
wave = {}
waveNumber = 0
initialSendDelay 	= 100	--OPTION: The number of ticks that must pass before units start being spawned
sendDelay 			= 40	--OPTION: The number of ticks between units being sent
parallelUnitSending = true	--OPTION: Setting this to true allows multiple units to spawn per tick at different spawns
endless 			= true	--OPTION: Controls whether the number of waves is finite
wavesToSurvive 		= 10	--OPTION: The number of waves performed if endless is false

--BUILDING MODE VARIABLES--
building 				= false
buildingActor			= nil
pathTiles 				= {}
pathBorderTiles 		= {}

--ENEMY VARIABLES--
enemies = {										--OPTION: Add/remove/edit units here
			{Cost = 1, 		Type = "e1"},
			{Cost = 1, 		Type = "dog"},
			{Cost = 3, 		Type = "zombie"},
			{Cost = 20, 	Type = "ant"},
			{Cost = 10,		Type = "jeep"},
			{Cost = 15, 	Type = "1tnk"},
			{Cost = 20, 	Type = "2tnk"},
			{Cost = 25, 	Type = "3tnk"},
			{Cost = 30, 	Type = "4tnk"},
			{Cost = 40, 	Type = "qtnk"}
		  }
